<?php
/**
 * Helper Functions untuk Perhitungan Penyusutan Aset
 * ACF EDU HUB - Inventory Management System
 */

/**
 * Hitung umur barang dalam tahun (dari tanggal perolehan sampai sekarang)
 * 
 * @param string $tanggal_perolehan Format: Y-m-d
 * @return float Umur dalam tahun (desimal)
 */
function hitungUmurBarang($tanggal_perolehan) {
    if (empty($tanggal_perolehan)) {
        return 0;
    }
    
    $tanggal_beli = new DateTime($tanggal_perolehan);
    $sekarang = new DateTime();
    $selisih = $tanggal_beli->diff($sekarang);
    
    // Hitung umur dalam tahun dengan desimal
    $umur_tahun = $selisih->y + ($selisih->m / 12) + ($selisih->d / 365);
    
    return round($umur_tahun, 2);
}

/**
 * Hitung penyusutan dengan metode garis lurus
 * 
 * @param float $harga_perolehan Harga awal pembelian
 * @param float $nilai_residu Nilai sisa di akhir umur ekonomis
 * @param int $umur_ekonomis Umur ekonomis dalam tahun
 * @param float $umur_barang Umur barang saat ini dalam tahun
 * @return array ['penyusutan_per_tahun', 'akumulasi_penyusutan', 'nilai_buku', 'persentase_penyusutan']
 */
function hitungPenyusutanGarisLurus($harga_perolehan, $nilai_residu, $umur_ekonomis, $umur_barang) {
    // Pastikan tidak ada pembagian dengan 0
    if ($umur_ekonomis <= 0) {
        return [
            'penyusutan_per_tahun' => 0,
            'akumulasi_penyusutan' => 0,
            'nilai_buku' => $harga_perolehan,
            'persentase_penyusutan' => 0,
            'persentase_tersisa' => 100
        ];
    }
    
    // Hitung penyusutan per tahun
    $penyusutan_per_tahun = ($harga_perolehan - $nilai_residu) / $umur_ekonomis;
    
    // Batasi umur barang tidak melebihi umur ekonomis
    $umur_untuk_hitung = min($umur_barang, $umur_ekonomis);
    
    // Hitung akumulasi penyusutan
    $akumulasi_penyusutan = $penyusutan_per_tahun * $umur_untuk_hitung;
    
    // Hitung nilai buku/wajar
    $nilai_buku = $harga_perolehan - $akumulasi_penyusutan;
    
    // Pastikan nilai buku tidak kurang dari nilai residu
    $nilai_buku = max($nilai_buku, $nilai_residu);
    
    // Hitung persentase penyusutan
    $persentase_penyusutan = ($harga_perolehan > 0) ? ($akumulasi_penyusutan / $harga_perolehan * 100) : 0;
    $persentase_tersisa = 100 - $persentase_penyusutan;
    
    return [
        'penyusutan_per_tahun' => round($penyusutan_per_tahun, 2),
        'akumulasi_penyusutan' => round($akumulasi_penyusutan, 2),
        'nilai_buku' => round($nilai_buku, 2),
        'persentase_penyusutan' => round($persentase_penyusutan, 2),
        'persentase_tersisa' => round($persentase_tersisa, 2)
    ];
}

/**
 * Format rupiah
 * 
 * @param float $angka
 * @return string
 */
function formatRupiah($angka) {
    return 'Rp ' . number_format($angka, 0, ',', '.');
}

/**
 * Get status kondisi barang berdasarkan persentase nilai tersisa
 * 
 * @param float $persentase_tersisa
 * @return array ['status', 'class', 'icon']
 */
function getStatusKondisi($persentase_tersisa) {
    if ($persentase_tersisa >= 80) {
        return [
            'status' => 'Sangat Baik',
            'class' => 'status-excellent',
            'icon' => 'fa-star',
            'color' => '#166534'
        ];
    } elseif ($persentase_tersisa >= 60) {
        return [
            'status' => 'Baik',
            'class' => 'status-good',
            'icon' => 'fa-check-circle',
            'color' => '#15803d'
        ];
    } elseif ($persentase_tersisa >= 40) {
        return [
            'status' => 'Cukup',
            'class' => 'status-fair',
            'icon' => 'fa-info-circle',
            'color' => '#ca8a04'
        ];
    } elseif ($persentase_tersisa >= 20) {
        return [
            'status' => 'Kurang',
            'class' => 'status-poor',
            'icon' => 'fa-exclamation-triangle',
            'color' => '#ea580c'
        ];
    } else {
        return [
            'status' => 'Perlu Penggantian',
            'class' => 'status-bad',
            'icon' => 'fa-times-circle',
            'color' => '#dc2626'
        ];
    }
}

/**
 * Estimasi harga jual/lelang (80-90% dari nilai buku)
 * 
 * @param float $nilai_buku
 * @return array ['min', 'max', 'rata_rata']
 */
function estimasiHargaJual($nilai_buku) {
    $min = $nilai_buku * 0.80;  // 80% dari nilai buku
    $max = $nilai_buku * 0.90;  // 90% dari nilai buku
    $rata_rata = ($min + $max) / 2;
    
    return [
        'min' => round($min, 2),
        'max' => round($max, 2),
        'rata_rata' => round($rata_rata, 2)
    ];
}

/**
 * Get rekomendasi tindakan berdasarkan kondisi aset
 * 
 * @param float $persentase_tersisa
 * @param float $umur_barang
 * @param int $umur_ekonomis
 * @return string
 */
function getRekomendasiTindakan($persentase_tersisa, $umur_barang, $umur_ekonomis) {
    $sisa_umur = $umur_ekonomis - $umur_barang;
    
    if ($persentase_tersisa >= 80) {
        return "Aset dalam kondisi sangat baik. Lanjutkan perawatan rutin.";
    } elseif ($persentase_tersisa >= 60) {
        return "Aset masih layak pakai. Perawatan berkala direkomendasikan.";
    } elseif ($persentase_tersisa >= 40) {
        return "Aset mendekati pertengahan umur ekonomis. Pertimbangkan untuk budgeting penggantian dalam $sisa_umur tahun.";
    } elseif ($persentase_tersisa >= 20) {
        return "Aset sudah tua. Siapkan anggaran penggantian segera. Sisa umur ekonomis: $sisa_umur tahun.";
    } else {
        return "Aset sudah melewati atau mendekati akhir umur ekonomis. Segera rencanakan penggantian atau pertimbangkan untuk dijual/dilelang.";
    }
}

/**
 * Hitung total nilai aset (untuk laporan)
 * 
 * @param array $aset_list Array of aset data
 * @return array Summary statistics
 */
function hitungTotalNilaiAset($aset_list) {
    $total_harga_perolehan = 0;
    $total_nilai_buku = 0;
    $total_penyusutan = 0;
    $jumlah_aset = count($aset_list);
    
    foreach ($aset_list as $aset) {
        $total_harga_perolehan += $aset['harga_perolehan'] ?? 0;
        $total_nilai_buku += $aset['nilai_buku'] ?? 0;
        $total_penyusutan += $aset['akumulasi_penyusutan'] ?? 0;
    }
    
    return [
        'jumlah_aset' => $jumlah_aset,
        'total_harga_perolehan' => $total_harga_perolehan,
        'total_nilai_buku' => $total_nilai_buku,
        'total_penyusutan' => $total_penyusutan,
        'rata_rata_persentase_tersisa' => ($total_harga_perolehan > 0) 
            ? round(($total_nilai_buku / $total_harga_perolehan * 100), 2) 
            : 0
    ];
}

/**
 * Format tampilan progress bar
 * 
 * @param float $persentase
 * @return string HTML progress bar
 */
function generateProgressBar($persentase, $label = '') {
    $persentase = max(0, min(100, $persentase)); // Batasi 0-100
    
    // Tentukan warna berdasarkan persentase
    if ($persentase >= 80) {
        $color = '#22c55e'; // Green
    } elseif ($persentase >= 60) {
        $color = '#84cc16'; // Lime
    } elseif ($persentase >= 40) {
        $color = '#eab308'; // Yellow
    } elseif ($persentase >= 20) {
        $color = '#f97316'; // Orange
    } else {
        $color = '#ef4444'; // Red
    }
    
    return "
    <div style='background: #f1f5f9; border-radius: 10px; overflow: hidden; height: 24px; position: relative;'>
        <div style='background: $color; width: {$persentase}%; height: 100%; transition: width 0.3s;'></div>
        <span style='position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); 
                     font-weight: 700; font-size: 12px; color: #1e293b;'>
            {$label} {$persentase}%
        </span>
    </div>
    ";
}
?>