<?php
session_start();

require_once __DIR__ . '/../../DataBase/koneksi.php';

// Cek login dan role
if (!isset($_SESSION['login']) || $_SESSION['login'] !== true) {
    header("Location: ../../index.html?error=session");
    exit;
}

if (!function_exists('checkRole')) {
    function checkRole($allowedRoles) {
        $userRole = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
        if (!in_array($userRole, $allowedRoles)) {
            header("Location: ../../index.html?error=unauthorized");
            exit;
        }
    }
}

checkRole(['A']); // Hanya Admin

$success = '';
$error = '';

// Cek apakah table settings ada, jika tidak buat
$checkTable = $conn->query("SHOW TABLES LIKE 'settings'");
if ($checkTable->num_rows == 0) {
    $createTable = "CREATE TABLE IF NOT EXISTS settings (
        id INT PRIMARY KEY AUTO_INCREMENT,
        setting_key VARCHAR(100) UNIQUE NOT NULL,
        setting_value TEXT,
        setting_type VARCHAR(50) DEFAULT 'text',
        description TEXT,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )";
    $conn->query($createTable);
    
    // Insert default settings
    $defaults = [
        ['app_name', 'Sistem Inventory ACF', 'text', 'Nama Aplikasi'],
        ['app_version', '1.0.0', 'text', 'Versi Aplikasi'],
        ['company_name', 'Yayasan Asih Cinta Fitri', 'text', 'Nama Perusahaan'],
        ['company_address', 'Jakarta, Indonesia', 'textarea', 'Alamat Perusahaan'],
        ['company_phone', '021-12345678', 'text', 'Telepon'],
        ['company_email', 'info@acf.or.id', 'email', 'Email'],
        ['items_per_page', '15', 'number', 'Jumlah item per halaman'],
        ['date_format', 'd/m/Y', 'text', 'Format Tanggal (PHP)'],
        ['currency_format', 'Rp', 'text', 'Format Mata Uang'],
        ['enable_notifications', '1', 'checkbox', 'Aktifkan Notifikasi'],
        ['maintenance_mode', '0', 'checkbox', 'Mode Maintenance']
    ];
    
    foreach ($defaults as $def) {
        $stmt = $conn->prepare("INSERT IGNORE INTO settings (setting_key, setting_value, setting_type, description) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("ssss", $def[0], $def[1], $def[2], $def[3]);
        $stmt->execute();
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $updated = 0;
    foreach ($_POST as $key => $value) {
        if ($key !== 'submit') {
            // Handle checkbox (jika tidak dicentang, valuenya tidak dikirim)
            $checkboxKeys = ['enable_notifications', 'maintenance_mode'];
            if (in_array($key, $checkboxKeys)) {
                $value = isset($_POST[$key]) ? '1' : '0';
            }
            
            $stmt = $conn->prepare("UPDATE settings SET setting_value = ? WHERE setting_key = ?");
            $stmt->bind_param("ss", $value, $key);
            if ($stmt->execute() && $stmt->affected_rows > 0) {
                $updated++;
            }
        }
    }
    
    if ($updated > 0) {
        $success = "Pengaturan berhasil disimpan! ($updated item diupdate)";
    } else {
        $error = "Tidak ada perubahan atau gagal menyimpan pengaturan.";
    }
}

// Get all settings
$settings = [];
$result = $conn->query("SELECT * FROM settings ORDER BY id");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $settings[$row['setting_key']] = $row;
    }
}

// Get system info
$totalUsers = $conn->query("SELECT COUNT(*) as total FROM users")->fetch_assoc()['total'] ?? 0;
$totalBarang = $conn->query("SELECT COUNT(*) as total FROM inventory")->fetch_assoc()['total'] ?? 0;
$dbSize = $conn->query("SELECT SUM(data_length + index_length) / 1024 / 1024 AS size FROM information_schema.TABLES WHERE table_schema = DATABASE()")->fetch_assoc()['size'] ?? 0;
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pengaturan Sistem</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.3/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            padding: 30px;
        }

        .page-header {
            margin-bottom: 30px;
        }

        .page-header h1 {
            color: #2d3748;
            font-size: 28px;
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .page-header p {
            color: #718096;
            font-size: 14px;
        }

        .header-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s;
        }

        .btn-secondary {
            background: #64748b;
            color: white;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
        }

        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border-left: 4px solid #10b981;
        }

        .alert-danger {
            background: #fee2e2;
            color: #991b1b;
            border-left: 4px solid #dc2626;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            color: white;
        }

        .stat-icon.purple { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .stat-icon.green { background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); }
        .stat-icon.blue { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }

        .stat-content h4 {
            font-size: 11px;
            color: #718096;
            text-transform: uppercase;
            margin-bottom: 5px;
        }

        .stat-content .value {
            font-size: 22px;
            font-weight: bold;
            color: #2d3748;
        }

        .settings-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 25px;
        }

        .card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }

        .card.full {
            grid-column: 1 / -1;
        }

        .card h3 {
            color: #2d3748;
            font-size: 18px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 8px;
            padding-bottom: 15px;
            border-bottom: 2px solid #e2e8f0;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            font-size: 13px;
            font-weight: 600;
            color: #334155;
            margin-bottom: 8px;
        }

        .form-group small {
            display: block;
            font-size: 12px;
            color: #64748b;
            margin-top: 4px;
        }

        .form-control {
            width: 100%;
            padding: 10px 14px;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            font-size: 14px;
        }

        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        textarea.form-control {
            resize: vertical;
            min-height: 80px;
        }

        .checkbox-wrapper {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .checkbox-wrapper input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }

        .form-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid #e2e8f0;
        }

        .info-list {
            list-style: none;
        }

        .info-list li {
            padding: 12px 0;
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .info-list li:last-child {
            border-bottom: none;
        }

        .info-list .label {
            color: #64748b;
            font-size: 14px;
        }

        .info-list .value {
            color: #2d3748;
            font-weight: 600;
            font-size: 14px;
        }

        @media (max-width: 1024px) {
            .settings-container {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 768px) {
            body {
                padding: 15px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="page-header">
        <h1><i class="fas fa-cog"></i> Pengaturan Sistem</h1>
        <p>Konfigurasi dan pengaturan sistem inventory ACF</p>
    </div>

    <div class="header-actions">
        <a href="javascript:history.back()" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Kembali
        </a>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <?php echo $success; ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i>
            <?php echo $error; ?>
        </div>
    <?php endif; ?>

    <!-- System Stats -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon purple">
                <i class="fas fa-users"></i>
            </div>
            <div class="stat-content">
                <h4>Total Users</h4>
                <div class="value"><?php echo $totalUsers; ?></div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon green">
                <i class="fas fa-boxes"></i>
            </div>
            <div class="stat-content">
                <h4>Total Barang</h4>
                <div class="value"><?php echo $totalBarang; ?></div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon blue">
                <i class="fas fa-database"></i>
            </div>
            <div class="stat-content">
                <h4>Ukuran Database</h4>
                <div class="value"><?php echo number_format($dbSize, 2); ?> MB</div>
            </div>
        </div>
    </div>

    <form method="POST" action="">
        <div class="settings-container">
            <!-- Informasi Aplikasi -->
            <div class="card">
                <h3><i class="fas fa-info-circle"></i> Informasi Aplikasi</h3>
                
                <div class="form-group">
                    <label>Nama Aplikasi</label>
                    <input type="text" name="app_name" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['app_name']['setting_value'] ?? 'Sistem Inventory ACF'); ?>">
                    <small>Nama aplikasi yang ditampilkan di header</small>
                </div>

                <div class="form-group">
                    <label>Versi Aplikasi</label>
                    <input type="text" name="app_version" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['app_version']['setting_value'] ?? '1.0.0'); ?>">
                    <small>Versi sistem saat ini</small>
                </div>
            </div>

            <!-- Informasi Perusahaan -->
            <div class="card">
                <h3><i class="fas fa-building"></i> Informasi Perusahaan</h3>
                
                <div class="form-group">
                    <label>Nama Perusahaan</label>
                    <input type="text" name="company_name" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['company_name']['setting_value'] ?? 'Yayasan Asih Cinta Fitri'); ?>">
                </div>

                <div class="form-group">
                    <label>Telepon</label>
                    <input type="text" name="company_phone" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['company_phone']['setting_value'] ?? ''); ?>">
                </div>

                <div class="form-group">
                    <label>Email</label>
                    <input type="email" name="company_email" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['company_email']['setting_value'] ?? ''); ?>">
                </div>
            </div>

            <!-- Alamat Perusahaan -->
            <div class="card full">
                <h3><i class="fas fa-map-marker-alt"></i> Alamat Perusahaan</h3>
                
                <div class="form-group">
                    <label>Alamat Lengkap</label>
                    <textarea name="company_address" class="form-control" rows="3"><?php echo htmlspecialchars($settings['company_address']['setting_value'] ?? ''); ?></textarea>
                    <small>Alamat lengkap yang akan muncul di laporan</small>
                </div>
            </div>

            <!-- Pengaturan Tampilan -->
            <div class="card">
                <h3><i class="fas fa-palette"></i> Pengaturan Tampilan</h3>
                
                <div class="form-group">
                    <label>Item per Halaman</label>
                    <input type="number" name="items_per_page" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['items_per_page']['setting_value'] ?? '15'); ?>" min="5" max="100">
                    <small>Jumlah item yang ditampilkan per halaman (5-100)</small>
                </div>

                <div class="form-group">
                    <label>Format Tanggal</label>
                    <input type="text" name="date_format" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['date_format']['setting_value'] ?? 'd/m/Y'); ?>">
                    <small>Format: d/m/Y, m/d/Y, Y-m-d</small>
                </div>

                <div class="form-group">
                    <label>Format Mata Uang</label>
                    <input type="text" name="currency_format" class="form-control" 
                           value="<?php echo htmlspecialchars($settings['currency_format']['setting_value'] ?? 'Rp'); ?>">
                    <small>Simbol mata uang: Rp, $, €, dll</small>
                </div>
            </div>

            <!-- Pengaturan Sistem -->
            <div class="card">
                <h3><i class="fas fa-sliders-h"></i> Pengaturan Sistem</h3>
                
                <div class="form-group">
                    <div class="checkbox-wrapper">
                        <input type="checkbox" name="enable_notifications" id="enable_notifications" 
                               value="1" <?php echo ($settings['enable_notifications']['setting_value'] ?? '1') == '1' ? 'checked' : ''; ?>>
                        <label for="enable_notifications" style="margin: 0;">Aktifkan Notifikasi</label>
                    </div>
                    <small>Tampilkan notifikasi sistem</small>
                </div>

                <div class="form-group">
                    <div class="checkbox-wrapper">
                        <input type="checkbox" name="maintenance_mode" id="maintenance_mode" 
                               value="1" <?php echo ($settings['maintenance_mode']['setting_value'] ?? '0') == '1' ? 'checked' : ''; ?>>
                        <label for="maintenance_mode" style="margin: 0;">Mode Maintenance</label>
                    </div>
                    <small style="color: #ef4444;">⚠️ Aktifkan saat melakukan maintenance</small>
                </div>
            </div>

            <!-- Informasi Sistem -->
            <div class="card full">
                <h3><i class="fas fa-server"></i> Informasi Sistem</h3>
                
                <ul class="info-list">
                    <li>
                        <span class="label">Versi PHP</span>
                        <span class="value"><?php echo phpversion(); ?></span>
                    </li>
                    <li>
                        <span class="label">Versi MySQL</span>
                        <span class="value"><?php echo $conn->server_info; ?></span>
                    </li>
                    <li>
                        <span class="label">Server</span>
                        <span class="value"><?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></span>
                    </li>
                    <li>
                        <span class="label">Sistem Operasi</span>
                        <span class="value"><?php echo PHP_OS; ?></span>
                    </li>
                    <li>
                        <span class="label">Max Upload Size</span>
                        <span class="value"><?php echo ini_get('upload_max_filesize'); ?></span>
                    </li>
                    <li>
                        <span class="label">Memory Limit</span>
                        <span class="value"><?php echo ini_get('memory_limit'); ?></span>
                    </li>
                </ul>
            </div>
        </div>

        <div class="card" style="margin-top: 25px;">
            <div class="form-actions">
                <button type="reset" class="btn btn-secondary">
                    <i class="fas fa-undo"></i> Reset
                </button>
                <button type="submit" name="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Simpan Pengaturan
                </button>
            </div>
        </div>
    </form>
</body>
</html>