<?php
session_start();

// Security check
if (!isset($_SESSION['login']) || $_SESSION['login'] !== true) {
    header("Location: ../../index.html?error=session");
    exit;
}

require_once __DIR__ . '/../../DataBase/koneksi.php';

// Helper function for currency format
if (!function_exists('formatRupiah')) {
    function formatRupiah($angka) {
        return 'Rp ' . number_format((float)$angka, 0, ',', '.');
    }
}

// Get and sanitize search input
$search = isset($_GET['search']) ? trim($_GET['search']) : '';

// Prepared statement to prevent SQL injection
// NOTE: Update column name dari 'kode_aset' ke 'kode_asset'
if (!empty($search)) {
    $sql = "SELECT * FROM inventory 
            WHERE (jenis_barang LIKE ? 
                OR kode_asset LIKE ? 
                OR kode_aset LIKE ?
                OR kantor LIKE ?) 
            ORDER BY id DESC";
    $stmt = $conn->prepare($sql);
    $searchParam = "%$search%";
    $stmt->bind_param("ssss", $searchParam, $searchParam, $searchParam, $searchParam);
} else {
    $sql = "SELECT * FROM inventory ORDER BY id DESC";
    $stmt = $conn->prepare($sql);
}

$stmt->execute();
$result = $stmt->get_result();
$total_items = $result->num_rows;
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventory List - ACF Edu Hub</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <style>
        :root {
            --primary: #8B4789;
            --primary-dark: #6d3669;
            --secondary: #64748b;
            --dark: #1e293b;
            --light: #f8fafc;
            --success: #059669;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body { 
            font-family: 'Inter', 'Segoe UI', sans-serif; 
            background: #f1f5f9;
            color: var(--dark);
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header { 
            background: white; 
            padding: 30px; 
            border-radius: 16px; 
            margin-bottom: 24px; 
            box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 4px solid var(--primary);
        }
        
        .header-title h1 {
            color: var(--dark);
            font-size: 24px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .header-title p {
            color: var(--secondary);
            font-size: 14px;
            margin-top: 4px;
        }
        
        .header-actions {
            display: flex;
            gap: 12px;
        }
        
        .btn { 
            padding: 10px 18px; 
            border-radius: 8px; 
            text-decoration: none; 
            display: inline-flex; 
            align-items: center; 
            gap: 8px; 
            font-weight: 600;
            font-size: 14px;
            transition: all 0.2s;
            border: none;
            cursor: pointer;
        }
        
        .btn:hover { transform: translateY(-1px); }
        
        .btn-back { background: var(--light); color: var(--secondary); border: 1px solid #e2e8f0; }
        .btn-back:hover { background: #e2e8f0; }
        
        .btn-print { background: var(--primary); color: white; }
        .btn-print:hover { background: var(--primary-dark); box-shadow: 0 4px 12px rgba(139, 71, 137, 0.3); }
        
        .search-container {
            background: white;
            padding: 20px;
            border-radius: 16px;
            margin-bottom: 24px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        
        .search-box {
            display: flex;
            gap: 12px;
        }
        
        .search-box input {
            flex: 1;
            padding: 12px 20px;
            border: 2px solid #f1f5f9;
            border-radius: 10px;
            font-size: 14px;
            background: #f8fafc;
            transition: all 0.3s;
        }
        
        .search-box input:focus {
            outline: none;
            border-color: var(--primary);
            background: white;
        }
        
        .btn-search {
            background: var(--dark);
            color: white;
            padding: 0 25px;
            border-radius: 10px;
            font-weight: 600;
        }

        .stats-bar {
            display: flex;
            justify-content: space-between;
            margin-top: 15px;
            font-size: 13px;
            color: var(--secondary);
        }
        
        .table-container { 
            background: white; 
            border-radius: 16px; 
            overflow: hidden; 
            box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1);
        }
        
        table { width: 100%; border-collapse: collapse; }
        
        th { 
            background: #f8fafc;
            color: var(--secondary); 
            padding: 16px; 
            text-align: left; 
            font-size: 12px;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1px;
            border-bottom: 2px solid #f1f5f9;
        }
        
        td { 
            padding: 16px; 
            border-bottom: 1px solid #f1f5f9; 
            font-size: 14px;
        }
        
        tr:last-child td { border-bottom: none; }
        tr:hover td { background: #fafafa; }
        
        /* Style untuk kode format baru PST.01.A01.01 */
        .kode-badge { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 8px 12px; 
            border-radius: 8px; 
            font-family: 'Courier New', monospace;
            font-weight: 700;
            font-size: 13px;
            letter-spacing: 1px;
            box-shadow: 0 2px 4px rgba(102, 126, 234, 0.2);
            display: inline-block;
        }
        
        /* Style untuk kode format lama ACF-PST-xxx */
        .kode-badge-old {
            background: #f1f5f9; 
            color: var(--secondary); 
            padding: 6px 10px; 
            border-radius: 6px; 
            font-family: 'Courier New', monospace;
            font-weight: 600;
            font-size: 12px;
            border: 1px solid #e2e8f0;
        }
        
        .kode-badge-none {
            background: #fef2f2;
            color: #991b1b;
            padding: 6px 10px;
            border-radius: 6px;
            font-size: 11px;
            font-weight: 600;
        }
        
        .status-badge {
            padding: 5px 12px;
            border-radius: 6px;
            font-size: 11px;
            font-weight: 700;
            text-transform: uppercase;
        }
        
        .status-aktif { background: #ecfdf5; color: #065f46; }
        .status-nonaktif { background: #fef2f2; color: #991b1b; }
        
        .price-text { color: var(--success); font-weight: 700; }

        .info-tag {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            font-size: 10px;
            background: #eff6ff;
            color: #1e40af;
            padding: 3px 8px;
            border-radius: 4px;
            margin-left: 8px;
        }

        @media (max-width: 768px) {
            .header { flex-direction: column; gap: 20px; align-items: flex-start; }
            .header-actions { width: 100%; }
            .btn { flex: 1; justify-content: center; }
            .table-container { overflow-x: auto; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="header-title">
                <h1><i class="fas fa-layer-group" style="color: var(--primary);"></i> Management Inventory</h1>
                <p>ACF Edu Hub Asset Tracking System</p>
            </div>
            
            <div class="header-actions">
                <a href="home_staff.php" class="btn btn-back">
                    <i class="fas fa-arrow-left"></i> Dashboard
                </a>
                <a href="../cetak_label_panel.php" class="btn btn-print">
                    <i class="fas fa-print"></i> Panel Cetak Label
                </a>
            </div>
        </div>

        <div class="search-container">
            <form method="GET" action="">
                <div class="search-box">
                    <input 
                        type="text" 
                        name="search" 
                        placeholder="Cari berdasarkan kode asset, nama barang, atau lokasi kantor..." 
                        value="<?= htmlspecialchars($search) ?>"
                    >
                    <button type="submit" class="btn btn-search">
                        <i class="fas fa-search"></i> Cari Data
                    </button>
                    <?php if (!empty($search)): ?>
                    <a href="?" class="btn" style="background:#fee2e2; color:#ef4444;">
                        <i class="fas fa-sync"></i> Reset
                    </a>
                    <?php endif; ?>
                </div>
            </form>
            
            <div class="stats-bar">
                <span><i class="fas fa-clipboard-list"></i> Menampilkan <b><?= $total_items ?></b> total aset</span>
                <?php if (!empty($search)): ?>
                    <span>Hasil pencarian untuk: <b>"<?= htmlspecialchars($search) ?>"</b></span>
                <?php endif; ?>
            </div>
        </div>

        <div class="table-container">
            <?php if ($result->num_rows > 0): ?>
            <table>
                <thead>
                    <tr>
                        <th>Kode Asset</th>
                        <th>Nama Barang</th>
                        <th>Lokasi Kantor</th>
                        <th>Harga Perolehan</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while($row = $result->fetch_assoc()): 
                        // Prioritaskan kode_asset (format baru), fallback ke kode_aset (lama)
                        $kode = '';
                        $kode_class = '';
                        
                        if (!empty($row['kode_asset'])) {
                            // Format baru: PST.01.A01.01
                            $kode = $row['kode_asset'];
                            $kode_class = 'kode-badge';
                        } elseif (!empty($row['kode_aset'])) {
                            // Format lama: ACF-PST-01-A-001
                            $kode = $row['kode_aset'];
                            $kode_class = 'kode-badge-old';
                        } else {
                            // Belum ada kode
                            $kode = 'NO-CODE';
                            $kode_class = 'kode-badge-none';
                        }
                    ?>
                    <tr>
                        <td>
                            <span class="<?= $kode_class ?>">
                                <?= htmlspecialchars($kode) ?>
                            </span>
                            <?php if (!empty($row['kode_asset']) && strpos($row['kode_asset'], '.') !== false): ?>
                                <span class="info-tag">
                                    <i class="fas fa-check-circle"></i> NEW
                                </span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div style="font-weight: 600; color: var(--dark);"><?= htmlspecialchars($row['jenis_barang']) ?></div>
                            <?php if (!empty($row['kategori'])): ?>
                                <small style="color: var(--secondary); font-size: 12px;">
                                    <?= htmlspecialchars($row['kategori']) ?>
                                </small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div style="display: flex; align-items: center; gap: 6px; color: var(--secondary);">
                                <i class="fas fa-building" style="font-size: 12px;"></i>
                                <?= htmlspecialchars($row['kantor']) ?>
                            </div>
                        </td>
                        <td><span class="price-text"><?= formatRupiah($row['harga_perolehan']) ?></span></td>
                        <td>
                            <span class="status-badge <?= $row['status'] === 'aktif' ? 'status-aktif' : 'status-nonaktif' ?>">
                                <?= htmlspecialchars($row['status']) ?>
                            </span>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
            <?php else: ?>
            <div style="text-align: center; padding: 60px; color: var(--secondary);">
                <i class="fas fa-folder-open" style="font-size: 48px; margin-bottom: 16px; opacity: 0.2;"></i>
                <p>Data tidak ditemukan atau masih kosong.</p>
                <?php if (!empty($search)): ?>
                    <p style="margin-top: 10px;">
                        <a href="?" style="color: var(--primary); text-decoration: none; font-weight: 600;">
                            <i class="fas fa-redo"></i> Tampilkan semua data
                        </a>
                    </p>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- Info Box -->
        <div style="background: #eff6ff; border-left: 4px solid #3b82f6; padding: 16px; border-radius: 8px; margin-top: 20px;">
            <div style="display: flex; align-items: start; gap: 12px;">
                <i class="fas fa-info-circle" style="color: #3b82f6; font-size: 20px; margin-top: 2px;"></i>
                <div style="font-size: 13px; color: #1e40af;">
                    <strong>Tentang Kode Asset:</strong><br>
                    • <span class="kode-badge" style="font-size: 11px; padding: 4px 8px;">PST.01.A01.01</span> = Format baru (sistem otomatis)<br>
                    • <span class="kode-badge-old" style="font-size: 11px; padding: 4px 8px;">ACF-PST-01-A-001</span> = Format lama<br>
                    • <span class="kode-badge-none" style="font-size: 11px; padding: 4px 8px;">NO-CODE</span> = Belum memiliki kode
                </div>
            </div>
        </div>
    </div>
</body>
</html>

<?php
$stmt->close();
$conn->close();
?>